<?php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json');
require_once '../inc_db.php';
$pdo = connectToDatabasePDO();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método não permitido']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
if (!is_array($input)) {
    http_response_code(400);
    echo json_encode(['error' => 'Payload inválido (JSON esperado)']);
    exit;
}

$required = ['balance_id', 'amount', 'type', 'type_person', 'available_balance', 'balance_data'];
foreach ($required as $field) {
    if (!isset($input[$field])) {
        http_response_code(400);
        echo json_encode(['error' => "Campo obrigatório ausente: $field"]);
        exit;
    }
}

$balanceId   = (int) $input['balance_id'];
$amount      = (float) $input['amount'];
$type        = trim($input['type']);
$typePerson  = trim($input['type_person']);
$available   = (float) $input['available_balance'];
$balance     = $input['balance_data'];
$now         = date('Y-m-d H:i:s');

// Valida dados do objeto de saldo
if (
    !isset($balance['status'], $balance['type_user'], $balance['points_used'], $balance['points_earned'], $balance['cpf']) ||
    (int)$balance['status'] !== 1 ||
    (int)$balance['type_user'] !== 1
) {
    http_response_code(403);
    echo json_encode(['error' => 'Saldo indisponível ou tipo de usuário inválido']);
    exit;
}

// Valida saldo disponível
if ($available < $amount) {
    http_response_code(400);
    echo json_encode(['error' => "Saldo insuficiente. Disponível: {$available}"]);
    exit;
}

try {
    
    // Debug: log detalhado
    /*
        error_log("### DEBUG RESGATE ###");
        error_log("balance_id: " . $balanceId);
        error_log("amount: " . $amount);
        error_log("available_balance: " . $available);
        error_log("type: " . $type);
        error_log("type_person: " . $typePerson);
        
        // Campos internos de balance_data
        error_log("---- balance_data ----");
        foreach ($balance as $k => $v) {
            error_log("$k: $v");
        }
    */
    /* Fim Debug Detalhado */
    
    
    if (!$pdo->inTransaction()) {
        $pdo->beginTransaction();
    }

    // Grava o resgate
    $stmt = $pdo->prepare("
        INSERT INTO loyalty_redemptions (
            type, cpf, type_person, amount, balance_id, created_at, updated_at
        ) VALUES (
            :type, :cpf, :type_person, :amount, :balance_id, :created_at, :updated_at
        )
    ");
    $stmt->execute([
        ':type' => $type,
        ':cpf' => $balance['cpf'],
        ':type_person' => $typePerson,
        ':amount' => $amount,
        ':balance_id' => $balanceId,
        ':created_at' => $now,
        ':updated_at' => $now
    ]);

    // Calcula novo saldo (sem alterar o anterior)
    $newPointsUsed = $amount;
    $newBalance = $balance['points_earned'] - $newPointsUsed;

    // Cria nova linha de saldo com pontos resgatados
    $stmt = $pdo->prepare("
        INSERT INTO loyalty_balances (
            customer_id, cpf, status, type_user, loyalty_id,
            points_earned, points_used, expires_at,
            created_at, updated_at 
        ) VALUES (
            :customer_id, :cpf, :status, :type_user, :loyalty_id,
            :points_earned, :points_used, :expires_at,
            :created_at, :updated_at 
        )
    ");
    $stmt->execute([
        ':customer_id' => $balance['customer_id'],
        ':cpf' => $balance['cpf'],
        ':status' => 1,
        ':type_user' => $balance['type_user'],
        ':loyalty_id' => $balance['loyalty_id'],
        ':points_earned' => $available,
        ':points_used' => $newPointsUsed,
        ':expires_at' => $balance['expires_at'],
        ':created_at' => $now,
        ':updated_at' => $now 
    ]);

    // Audita a ação
    $stmt = $pdo->prepare("
        INSERT INTO loyalty_audit_logs (
            action, table_name, record_id, cpf, user_id, ip_address, user_agent,
            payload_before, payload_after, created_at
        ) VALUES (
            'REDEEM', 'loyalty_balances', :record_id, :cpf, NULL, :ip, :agent,
            :before, :after, :created_at
        )
    ");
    
    $newBalancePayload = [
        'customer_id'    => $balance['customer_id'],
        'cpf'            => $balance['cpf'],
        'status'         => 1,
        'type_user'      => $balance['type_user'],
        'loyalty_id'     => $balance['loyalty_id'],
        'points_earned'  => $available,
        'points_used'    => $newPointsUsed,
        'expires_at'     => $balance['expires_at'],
        'created_at'     => $now,
        'updated_at'     => $now
    ];
    
    $stmt->execute([
        ':record_id'     => $balanceId,
        ':cpf'           => $balance['cpf'],
        ':ip'            => $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0',
        ':agent'         => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
        ':before'        => json_encode($balance),
        ':after'         => json_encode($newBalancePayload),
        ':created_at'    => $now
    ]);


    $pdo->commit();

    echo json_encode([
        'success' => true,
        'cpf' => $balance['cpf'],
        'resgatado' => $amount,
        'saldo_restante' => $newBalance
    ]);

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    http_response_code(500);
    echo json_encode([
        'error' => 'Falha no resgate',
        'details' => $e->getMessage()
    ]);
}
