<?php
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json');
require_once '../inc_db.php';
require_once 'audit_log.php';


function calcularConsumerRepass(array $program, float $amount): float {
    $consumerRepass = 0.00;

    if (!empty($program['customers_enabled']) && $program['customers_enabled']) {
        $type = $program['customers_reward_type'] ?? '';
        $value = floatval($program['customers_reward_value'] ?? 0);
        $minPurchase = floatval($program['customers_min_purchase'] ?? 0);

        if ($type === 'TPV' && $amount >= $minPurchase) {
            $consumerRepass = $amount * ($value / 100);
        } elseif ($type === 'Fixed') {
            $consumerRepass = $value;
        } elseif ($type === 'Points') {
            $consumerRepass = $amount * $value * 10; // Ex: 1 real = 10 pontos
        }
    }

    return round($consumerRepass, 2);
}


function calcularAffiliateRepass(array $program, float $amount): float {
    $affiliateRepass = 0.00;

    if (!empty($program['affiliates_enabled']) && $program['affiliates_enabled']) {
        $type = $program['affiliates_commission_type'] ?? '';
        $value = floatval($program['affiliates_commission_value'] ?? 0);
        $minSale = floatval($program['affiliates_minimum_sale'] ?? 0);

        if ($type === 'TPV' && $amount >= $minSale) {
            $affiliateRepass = $amount * ($value / 100);
        } elseif ($type === 'Fixed') {
            $affiliateRepass = $value;
        }
    }

    return round($affiliateRepass, 2);
}




try {
    $rawInput = file_get_contents('php://input');
    error_log("[INÍCIO] Recebido JSON bruto: $rawInput");

    $json = json_decode($rawInput, true);
    if (!$json) {
        error_log("❌ Erro ao decodificar JSON: " . json_last_error_msg());
        echo json_encode(['success' => false, 'message' => 'JSON inválido.']);
        exit;
    }

    // === Sanitização e logging de dados principais ===
    $cpf = preg_replace('/\D/', '', $json['consumer_cpf'] ?? '');
    $cnpj = preg_replace('/\D/', '', $json['merchant_cnpj'] ?? '');
    $amountRaw = $json['amount'] ?? 0;

    // Garantia de ponto flutuante válido
    $amount = is_numeric($amountRaw) ? (float)$amountRaw : 0.0;

    error_log("📌 CPF: $cpf | CNPJ: $cnpj | Valor recebido: $amountRaw => Convertido: $amount");

    $pdo = connectToDatabasePDO();

    // === Busca do programa ativo ===
    $stmt = $pdo->prepare("SELECT id FROM loyalty_programs WHERE cnpj = ? AND status = 'ACTIVE' AND CURRENT_DATE BETWEEN start_date AND end_date ORDER BY end_date DESC LIMIT 1");
    $stmt->execute([$cnpj]);
    $loyalty_id = $stmt->fetchColumn();

    if (!$loyalty_id) {
        error_log("⚠️ Programa de fidelidade não encontrado para CNPJ $cnpj");
        echo json_encode(['success' => false, 'message' => 'Programa de fidelidade não encontrado para o CNPJ.']);
        exit;
    }

    $stmt = $pdo->prepare("SELECT * FROM loyalty_programs WHERE id = ?");
    $stmt->execute([$loyalty_id]);
    $program = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$program) {
        error_log("❌ Programa de fidelidade com ID $loyalty_id não localizado.");
        echo json_encode(['success' => false, 'message' => 'Erro ao buscar programa.']);
        exit;
    }


    // === Cálculo dos repasses ===
    $consumer_repass = calcularConsumerRepass($program, $amount);
    $affiliate_repass = calcularAffiliateRepass($program, $amount);




    // === Upload do comprovante ===
    $path_document = '';
    if (!empty($_FILES['document']) && $_FILES['document']['error'] === UPLOAD_ERR_OK) {
        $ext = pathinfo($_FILES['document']['name'], PATHINFO_EXTENSION);
        $filename = date('Y-m-d__H-i-s') . "." . $ext;
        $upload_dir = __DIR__ . "/../uploads/{$loyalty_id}/{$cpf}/";
        if (!is_dir($upload_dir)) mkdir($upload_dir, 0777, true);
        $path_document = "uploads/{$loyalty_id}/{$cpf}/{$filename}";
        move_uploaded_file($_FILES['document']['tmp_name'], $upload_dir . $filename);
        error_log("📎 Documento salvo em: $path_document");
    }

    // === Inserção na base ===
    $stmt = $pdo->prepare("INSERT INTO loyalty_purchases (
        loyalty_id, beneficiary_account_id, transaction_type, amount, tipo_dispositivo, email, card_last4,
        authorization_code, acquirer_id, consumer_repass, terminal_id, affiliate_repass,
        description, transaction_date, status, path_document, document_text, merchant_razao_social,
        merchant_cnpj, merchant_endereco, nf_numero, nf_serie, nf_emissao, nf_protocolo_autorizacao,
        nf_chave_acesso, consumer_cpf, modalidade, bandeira, metodo_pagamento, numero_parcelas,
        created_at
    ) VALUES (
        :loyalty_id, :beneficiary_account_id, :transaction_type, :amount, :tipo_dispositivo, :email, :card_last4,
        :authorization_code, :acquirer_id, :consumer_repass, :terminal_id, :affiliate_repass,
        :description, :transaction_date, :status, :path_document, :document_text, :merchant_razao_social,
        :merchant_cnpj, :merchant_endereco, :nf_numero, :nf_serie, :nf_emissao, :nf_protocolo_autorizacao,
        :nf_chave_acesso, :consumer_cpf, :modalidade, :bandeira, :metodo_pagamento, :numero_parcelas,
        NOW()
    )");

    $stmt->execute([
        ':loyalty_id' => $loyalty_id,
        ':beneficiary_account_id' => $json['beneficiary_account_id'] ?? null,
        ':transaction_type' => $json['transaction_type'] ?? '',
        ':amount' => $amount,
        ':tipo_dispositivo' => $json['tipo_dispositivo'] ?? '',
        ':email' => $json['email'] ?? '',
        ':card_last4' => $json['card_last4'] ?? '',
        ':authorization_code' => $json['authorization_code'] ?? '',
        ':acquirer_id' => $json['acquirer_id'] ?? '',
        ':consumer_repass' => $consumer_repass,
        ':terminal_id' => $json['terminal_id'] ?? '',
        ':affiliate_repass' => $affiliate_repass,
        ':description' => $json['description'] ?? '',
        ':transaction_date' => $json['transaction_date'] ?? date('Y-m-d H:i:s'),
        ':status' => $json['status'] ?? 'pendente',
        ':path_document' => $path_document,
        ':document_text' => $json['document_text'] ?? '',
        ':merchant_razao_social' => $json['merchant_razao_social'] ?? '',
        ':merchant_cnpj' => $cnpj,
        ':merchant_endereco' => $json['merchant_endereco'] ?? '',
        ':nf_numero' => $json['nf_numero'] ?? '',
        ':nf_serie' => $json['nf_serie'] ?? '',
        ':nf_emissao' => $json['nf_emissao'] ?? null,
        ':nf_protocolo_autorizacao' => $json['nf_protocolo_autorizacao'] ?? '',
        ':nf_chave_acesso' => $json['nf_chave_acesso'] ?? '',
        ':consumer_cpf' => $cpf,
        ':modalidade' => $json['modalidade'] ?? '',
        ':bandeira' => $json['bandeira'] ?? '',
        ':metodo_pagamento' => $json['metodo_pagamento'] ?? '',
        ':numero_parcelas' => $json['numero_parcelas'] ?? 1
    ]);

    $purchase_id = $pdo->lastInsertId();

    // === Registro de auditoria e debug final ===
    error_log("✅ Compra registrada ID: $purchase_id | Valor: $amount | Repasses => Cliente: $consumer_repass | Afiliado: $affiliate_repass");

    salvar_auditoria($pdo, [
        'action' => 'insert',
        'table' => 'loyalty_purchases',
        'record_id' => $purchase_id,
        'cpf' => $cpf,
        'payload_before' => json_encode([
            'consumer_repass' => $consumer_repass,
            'affiliate_repass' => $affiliate_repass,
            'loyalty_id' => $loyalty_id,
            'cnpj' => $cnpj,
            'cpf' => $cpf,
            'path_document' => $path_document
        ]),
        'payload_after' => json_encode($json)
    ]);

    echo json_encode(['success' => true, 'message' => 'Compra registrada com sucesso.', 'id' => $purchase_id]);

} catch (Exception $e) {
    error_log("❌ Exceção capturada: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Erro interno.', 'error' => $e->getMessage()]);
}
?>
