<?php
header('Content-Type: application/json');

// Permite apenas requisições POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método não permitido. Use POST.']);
    exit;
}

// Inclui a função de conexão
require_once '../inc_db.php';

// Lê e decodifica o JSON do corpo da requisição
$rawInput = file_get_contents('php://input');
$data = json_decode($rawInput, true);

// Validação do JSON
if (!is_array($data)) {
    http_response_code(400);
    echo json_encode(['error' => 'Requisição malformada. Envie um JSON válido.']);
    exit;
}

// Sanitização dos dados recebidos
$cpf  = preg_replace('/\D/', '', $data['cpf'] ?? '');
$cnpj = preg_replace('/\D/', '', $data['cnpj'] ?? '');

// Validação de formatos
if (strlen($cpf) !== 11 || strlen($cnpj) !== 14) {
    http_response_code(422);
    echo json_encode(['error' => 'CPF ou CNPJ inválido.']);
    exit;
}

// Conecta ao banco usando a função existente
$pdo = connectToDatabasePDO();

//DATE_FORMAT(BAL.updated_at, '%d/%m/%Y %H:%i:%s') AS ultima_movimentacao

// Consulta SQL
    $sql = "
    SELECT 
        LP.id AS program_id,
        LP.program_name,
        ACC.full_name,
        BAL.id AS balance_id,
        ACC.account_id,
        BAL.expires_at,
    
        -- Saldo atual (só 1 linha, sem SUM)
        CASE 
            WHEN DATEDIFF(NOW(), BAL.updated_at) > 90 THEN 0.00
            ELSE CASE 
                WHEN BAL.expires_at >= CURRENT_DATE THEN BAL.current_balance
                ELSE 0.00
            END
        END AS current_balance,
    
        -- Expira amanhã
        CASE 
            WHEN DATE(BAL.expires_at) = DATE_ADD(CURDATE(), INTERVAL 1 DAY) THEN BAL.current_balance
            ELSE 0.00
        END AS expira_amanha,
    
        -- Expira em até 7 dias
        CASE 
            WHEN DATE(BAL.expires_at) > DATE_ADD(CURDATE(), INTERVAL 1 DAY)
             AND DATE(BAL.expires_at) <= DATE_ADD(CURDATE(), INTERVAL 7 DAY)
            THEN BAL.current_balance
            ELSE 0.00
        END AS expira_7dias,
    
        DATE_FORMAT(BAL.updated_at, '%d/%m/%Y') AS updated_at
    
    FROM loyalty_programs AS LP
    
    -- Subquery apenas com o último saldo
    INNER JOIN (
        SELECT *
        FROM loyalty_balances
        WHERE cpf = :cpf
        ORDER BY updated_at DESC
        LIMIT 1
    ) AS BAL ON BAL.loyalty_id = LP.id
    
    LEFT JOIN update54_MUYTO_v2.account_personal AS ACC 
        ON ACC.cpf = BAL.cpf
    
    WHERE 
        LP.status = 'ACTIVE'
        AND CURRENT_DATE BETWEEN LP.start_date AND LP.end_date
        AND LP.cnpj = :cnpj
    
    ORDER BY LP.duration DESC
    LIMIT 1;


";

/*error_log("Consulta executada com parâmetros:");
error_log("CNPJ: " . $cnpj);
error_log("CPF: " . $cpf);
error_log("   >> SQL: " . $sql . "  << ");
*/


try {
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':cnpj' => $cnpj,
        ':cpf'  => $cpf
    ]);

    $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

    
    //error_log("Debugging \$result: " . print_r($result, true));
    
    if (count($result) === 0) {
        http_response_code(200);
        echo json_encode([
            'success' => false,
            'message' => 'Nenhum saldo encontrado para o CPF informado.'
        ]);
        exit;
    }



    echo json_encode([
        'success' => true,
        'data' => $result
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Erro na consulta: ' . $e->getMessage()]);
}
