<?php
// Inclua o arquivo de conexão com o banco de dados
require_once 'inc_db.php';

// Defina o cabeçalho para retornar conteúdo JSON
header('Content-Type: application/json');

// Exemplo de uso:
try {
    // Estabelece conexão com o banco de dados usando a função incluída
    $pdo = connectToDatabasePDO();
    error_log("Conexão com o banco de dados estabelecida.");

    $beneficiary_id = 1;
    $loyalty_id = 1; // Ajuste conforme o seu cenário
    $bonus_value = 12.00; // Valor do bônus a ser utilizado

    if (redeemBonus($pdo, $beneficiary_id, $loyalty_id, $bonus_value)) {
        echo "Bônus resgatado com sucesso!";
    }
} catch (Exception $ex) {
    echo "Erro: " . $ex->getMessage();
}


////////////////////////////////////////////////////////////////////////////////////////


function redeemBonus(PDO $pdo, int $beneficiary_id, int $loyalty_id, float $bonus_value): bool {
    try {
        // Inicia a transação
        $pdo->beginTransaction();

        // Consulta o total de bônus disponível para o cliente
        $stmtTotal = $pdo->prepare("
            SELECT IFNULL(SUM(remaining_amount), 0) as total 
            FROM loyalty_transactions 
            WHERE beneficiary_account_id = ? 
              AND loyalty_id = ? 
              AND transaction_type = 'EARN' 
              AND available_at <= NOW() 
              AND remaining_amount > 0
        ");
        $stmtTotal->execute([$beneficiary_id, $loyalty_id]);
        $row = $stmtTotal->fetch(PDO::FETCH_ASSOC);
        $total_available = (float)$row['total'];

        if ($total_available < $bonus_value) {
            throw new Exception("Bônus insuficiente para resgate");
        }

        // Busca os registros EARN disponíveis, ordenados pela data de liberação
        $stmtSelect = $pdo->prepare("
            SELECT id, remaining_amount 
            FROM loyalty_transactions 
            WHERE beneficiary_account_id = ? 
              AND loyalty_id = ? 
              AND transaction_type = 'EARN' 
              AND available_at <= NOW() 
              AND remaining_amount > 0 
            ORDER BY available_at ASC
        ");
        $stmtSelect->execute([$beneficiary_id, $loyalty_id]);
        $earnRecords = $stmtSelect->fetchAll(PDO::FETCH_ASSOC);

        $remaining_to_redeem = $bonus_value;

        // Prepara a query para atualizar o saldo dos registros EARN
        $stmtUpdate = $pdo->prepare("
            UPDATE loyalty_transactions 
            SET remaining_amount = remaining_amount - ? 
            WHERE id = ?
        ");

        // Prepara a query para inserir a transação REDEEM
        $stmtInsert = $pdo->prepare("
            INSERT INTO loyalty_transactions (
                beneficiary_account_id, 
                loyalty_id, 
                transaction_type, 
                amount, 
                parent_transaction_id, 
                description, 
                transaction_date
            ) VALUES (?, ?, 'REDEEM', ?, ?, ?, NOW())
        ");

        // Itera sobre os registros EARN disponíveis
        foreach ($earnRecords as $record) {
            if ($remaining_to_redeem <= 0) {
                break;
            }
            $earn_id = $record['id'];
            $available = (float)$record['remaining_amount'];

            if ($available <= $remaining_to_redeem) {
                // Utiliza o valor total do registro EARN
                $redeem_amount = $available;
                $stmtUpdate->execute([$redeem_amount, $earn_id]);

                $description = "Resgate total do bônus da transação {$earn_id} (R\$ " . number_format($redeem_amount, 2) . ")";
                $stmtInsert->execute([$beneficiary_id, $loyalty_id, $redeem_amount, $earn_id, $description]);

                $remaining_to_redeem -= $redeem_amount;
            } else {
                // Utiliza parcialmente o registro EARN
                $redeem_amount = $remaining_to_redeem;
                $stmtUpdate->execute([$redeem_amount, $earn_id]);

                $description = "Resgate parcial do bônus da transação {$earn_id} (R\$ " . number_format($redeem_amount, 2) . ")";
                $stmtInsert->execute([$beneficiary_id, $loyalty_id, $redeem_amount, $earn_id, $description]);

                $remaining_to_redeem = 0;
                break;
            }
        }

        if ($remaining_to_redeem > 0) {
            // Se ainda houver valor não resgatado, ocorre um erro
            $pdo->rollBack();
            throw new Exception("Erro no resgate do bônus: valor remanescente não resgatado");
        }

        // Confirma a transação
        $pdo->commit();
        return true;
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        // Trate ou registre o erro conforme necessário
        throw $e;
    }
}



/*

SELECT 
  lp.id,
  lp.program_name,
  lp.program_type,
  lp.start_date,
  lp.end_date,
  lp.status,
  lp.store_id,
  lt.beneficiary_account_id,
  COUNT(*) AS total_transacoes,
  SUM(CASE 
        WHEN TIMESTAMPDIFF(DAY, lt.transaction_date, NOW()) >= 8 
          THEN lt.amount 
        ELSE 0 
      END) AS total_faturado,
  SUM(CASE 
        WHEN TIMESTAMPDIFF(DAY, lt.transaction_date, NOW()) >= 8 
          THEN lt.remaining_amount 
        ELSE 0 
      END) AS total_bonus_disponivel,
  SUM(CASE 
        WHEN TIMESTAMPDIFF(DAY, lt.transaction_date, NOW()) < 8 
          THEN lt.amount 
        ELSE 0 
      END) AS total_bonus_em_processamento,
  ROUND(
    SUM(
      CASE 
          WHEN lp.customers_reward_type = 'FIXED' THEN lp.customers_reward_value
          WHEN lp.customers_reward_type = 'POINTS' THEN lp.customers_reward_value
          WHEN lp.customers_reward_type = 'PERCENT' THEN (lt.amount / 100) * lp.customers_reward_value
          ELSE 0
      END
    ), 2) AS total_customers_reward,
  ROUND(
    SUM(
      CASE 
          WHEN lp.affiliates_enabled = 1 THEN
              CASE 
                  WHEN lp.affiliates_commission_type = 'TPV' THEN (lt.amount / 100) * lp.affiliates_commission_value
                  WHEN lp.affiliates_commission_type = 'FIXED' THEN lp.affiliates_commission_value
                  ELSE 0
              END
          ELSE 0
      END
    ), 2) AS total_affiliates_commission
FROM 
  loyalty_transactions lt
  INNER JOIN loyalty_programs lp ON lt.loyalty_id = lp.id
WHERE 
  lt.transaction_type = 'EARN'
  AND lp.status = 'ACTIVE'
  AND lt.beneficiary_account_id = 1
  AND NOW() BETWEEN lp.start_date AND lp.end_date
GROUP BY 
  lp.id;



*/

?>
