<?php
/**
 * API de Fidelidade
 *
 * Campos esperados (via JSON no corpo da requisição):
 * ----------------------------------------------------
 *  action: string – define a operação a ser executada.
 *
 * AÇÕES IMPLEMENTADAS NESTE ARQUIVO:
 * ----------------------------------------------------
 *
 *  1) "EARN" – Cadastrar créditos (EARN)
 *     Parâmetros obrigatórios:
 *       - account_id (int)
 *       - loyalty_program_id (int)
 *       - amount (decimal)
 *     Parâmetros opcionais:
 *       - description (string)
 *
 *  2) "DETAIL" – Detalhar uma transação
 *     Parâmetros obrigatórios:
 *       - transaction_id (int)
 *
 *  3) "LIST_PROGRAMS" – Listar programas de fidelidade de uma conta
 *     Parâmetros obrigatórios:
 *       - account_id (int)
 *
 *  4) "CREATE_PROGRAM" – Criar um novo programa de fidelidade
 *     Parâmetros obrigatórios:
 *       - account_id (int)  // dono do programa
 *       - store_id (int)
 *       - program_type (string) [points, miles, cashback, etc.]
 *       - program_name (string)
 *     Parâmetros opcionais:
 *       - program_description, start_date, end_date, status, is_indefinite,
 *         affiliates_enabled, affiliates_commission_type, affiliates_commission_value,
 *         affiliates_minimum_sale, customers_enabled, customers_reward_type,
 *         customers_reward_value, customers_min_purchase, points_enabled, points_ratio,
 *         points_bonus, points_extra
 *
 *  5) "DETAIL_PROGRAM" – Obter detalhes completos de um programa
 *     Parâmetros obrigatórios:
 *       - program_id (int)
 *
 *  6) "UPDATE_PROGRAM" – Atualizar dados de um programa
 *     Parâmetros obrigatórios:
 *       - program_id (int)
 *     Parâmetros opcionais (enviados no JSON):
 *       - program_name, program_description, start_date, end_date, program_type,
 *         store_id, status, etc.
 *
 *  7) "DELETE_PROGRAM" – Exclusão lógica do programa (status = 'DELETED')
 *     Parâmetros obrigatórios:
 *       - program_id (int)
 *
 *  8) "GET_PROGRAM_DETAIL" – Obter detalhes completos de um programa de fidelidade
 *     Parâmetros obrigatórios:
 *       - program_id (int)
 *
 *  9) "GET_TRANSACTION_ITEMS" – Obter itens/detalhes de uma transação
 *     Parâmetros obrigatórios:
 *       - transaction_id (int)
 *
 * 10) "FILTER_TRANSACTIONS" – Filtrar transações (com parâmetros opcionais)
 *     Parâmetros obrigatórios:
 *       - account_id (int)
 *     Parâmetros opcionais:
 *       - days (int), consumer_cpf, consumer_cnpj, consumer_nome_razao, transaction_date
 *
 * 11) "LIST_FILTER_TRANSACTIONS" – Lista transações agregadas por conta/programa
 *     Parâmetros obrigatórios:
 *       - account_id (int)
 *     Parâmetros opcionais:
 *       - cpf, cnpj, razao_social, programa, data (formato dd/mm/yy), page (int, paginação)
 *
 * 12) "GET_PROGRAM_TRANSACTION_DETAIL" – Lista transações de um programa específico
 *     Parâmetros obrigatórios:
 *       - program_id (int)
 *     Parâmetros opcionais:
 *       - type_transaction (int) // se tiver mapeado no banco
 *       - beneficiary_account_id (int)
 *
 * 13) "LIST_CALCULATE_BONUS_MULTIPLIQUE" – Calcula dados de bônus/afiliados
 *     Parâmetros obrigatórios:
 *       - account_id (int)
 *     Parâmetros opcionais:
 *       - store_id, start_date, end_date (para filtrar a query)
 *
 */

require_once '../inc_db.php';
header('Content-Type: application/json');

/**
 * Obtém IP do usuário (para logs)
 */
function getUserIP() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        return $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        return $_SERVER['REMOTE_ADDR'];
    }
}

/**
 * Registra o log de chamadas da API.
 *
 * @param PDO    $pdo             Conexão com o banco de dados.
 * @param int    $objectId        ID do objeto relacionado (se aplicável).
 * @param string $requestPayload  JSON com o payload da requisição.
 * @param string $responsePayload JSON com a resposta gerada.
 * @param int    $statusCode      Código HTTP da resposta.
 */
function logApiCall($pdo, $objectId, $requestPayload, $responsePayload, $statusCode) {
    try {
        $ip_address = getUserIP();
        $stmt = $pdo->prepare("
            INSERT INTO account_track (
                id, request_payload, ip_address, response_payload, status_code, created_at
            ) VALUES (
                :object_id, :request_payload, :ip_address, :response_payload, :status_code, NOW()
            )
        ");
        $stmt->execute([
            ':object_id'        => $objectId,
            ':request_payload'  => $requestPayload,
            ':ip_address'       => $ip_address,
            ':response_payload' => $responsePayload,
            ':status_code'      => $statusCode
        ]);
    } catch (Exception $e) {
        error_log("Erro ao registrar log da API: " . $e->getMessage());
    }
}

/**
 * Envia a resposta em JSON e encerra o script.
 *
 * @param array $response Array com a resposta a ser enviada.
 */
function sendResponse($response) {
    echo json_encode($response);
    exit;
}

// --------------------------------------------------------
// 1. Ler o JSON enviado via POST e validar campo 'action'
// --------------------------------------------------------
$input = file_get_contents('php://input');
$data  = json_decode($input, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    $error_response = ['error' => 'JSON inválido'];
    echo json_encode($error_response);
    exit;
}

if (!isset($data['action'])) {
    http_response_code(400);
    $error_response = ['error' => 'Campo "action" é obrigatório.'];
    echo json_encode($error_response);
    exit;
}

$action = strtoupper(trim($data['action']));

// ----------------------------------
// 2. Conectar ao banco de dados PDO
// ----------------------------------
try {
    $pdo = connectToDatabasePDO();
} catch (Exception $e) {
    http_response_code(500);
    $error_response = [
        'error'   => 'Erro ao conectar ao banco de dados',
        'details' => $e->getMessage()
    ];
    echo json_encode($error_response);
    exit;
}

// ------------------------------------------------------
// 3. Executar a ação solicitada no switch($action)
// ------------------------------------------------------
switch ($action) {

    /**
     * 1) EARN - Cadastrar créditos
     */
    case 'EARN':
        try {
            if (!isset($data['account_id'], $data['loyalty_program_id'], $data['amount'])) {
                throw new Exception('Campos "account_id", "loyalty_program_id" e "amount" são obrigatórios para EARN.');
            }
            $account_id         = (int)$data['account_id'];
            $loyalty_program_id = (int)$data['loyalty_program_id'];
            $amount             = (float)$data['amount'];
            $description        = isset($data['description']) ? $data['description'] : '';

            // Exemplo de gravação de saldo (simples)
            $stmt = $pdo->prepare("
                SELECT id, balance 
                FROM loyalty_balance 
                WHERE account_id = ? AND loyalty_program_id = ?
            ");
            $stmt->execute([$account_id, $loyalty_program_id]);
            $balanceRecord = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($balanceRecord) {
                // Já existe saldo, atualizar
                $loyalty_balance_id = $balanceRecord['id'];
                $newBalance         = $balanceRecord['balance'] + $amount;

                $stmt = $pdo->prepare("
                    UPDATE loyalty_balance 
                       SET balance = ?, updated_at = NOW() 
                     WHERE id = ?
                ");
                $stmt->execute([$newBalance, $loyalty_balance_id]);
            } else {
                // Não existe saldo, inserir
                $stmt = $pdo->prepare("
                    INSERT INTO loyalty_balance (loyalty_program_id, account_id, balance) 
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$loyalty_program_id, $account_id, $amount]);
                $loyalty_balance_id = $pdo->lastInsertId();
                $newBalance         = $amount;
            }

            // Registrar transação de crédito
            $stmt = $pdo->prepare("
                INSERT INTO loyalty_transactions (loyalty_balance_id, transaction_type, amount, description)
                VALUES (?, 'EARN', ?, ?)
            ");
            $stmt->execute([$loyalty_balance_id, $amount, $description]);

            $response = [
                'success'     => true,
                'message'     => 'Crédito cadastrado com sucesso',
                'new_balance' => $newBalance
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;
        
        // =========================================================
        // 3. "EXTRACT" – Obter extrato dos últimos X dias
        // =========================================================
        case 'EXTRACT':
            try {
                if (!isset($data['account_id'])) {
                    throw new Exception('Campo account_id é obrigatório para EXTRACT');
                }
                $account_id = (int)$data['account_id'];
                $days = isset($data['days']) ? (int)$data['days'] : 7;
                $stmt = $pdo->prepare("
                
                SELECT lt.*
                    FROM loyalty_transactions lt, loyalty_programs lp
                    WHERE lt.loyalty_id = lp.id
    				AND lp.account_id = ? AND lt.transaction_date >= DATE_SUB(NOW(), INTERVAL ? DAY)
                    ORDER BY lt.transaction_date DESC
    
                ");
                $stmt->execute([$account_id, $days]);
                $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $response = [
                    'success'      => true,
                    'transactions' => $transactions
                ];
                logApiCall($pdo, 0, $input, json_encode($response), 200);
                sendResponse($response);
            } catch (Exception $e) {
                http_response_code(500);
                $error_response = ['error' => $e->getMessage()];
                logApiCall($pdo, 0, $input, json_encode($error_response), 500);
                sendResponse($error_response);
            }
            break;

        
        

    /**
     * 2) DETAIL - Detalhar uma transação
     */
    case 'DETAIL':
        try {
            if (!isset($data['transaction_id'])) {
                throw new Exception('Campo "transaction_id" é obrigatório para DETAIL.');
            }
            $transaction_id = (int)$data['transaction_id'];

            $stmt = $pdo->prepare("SELECT * FROM loyalty_transactions WHERE id = ?");
            $stmt->execute([$transaction_id]);
            $transaction = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$transaction) {
                throw new Exception('Transação não encontrada.');
            }
            $response = [
                'success'     => true,
                'transaction' => $transaction
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 3) LIST_PROGRAMS - Listar programas de fidelidade de uma conta
     */
    case 'LIST_PROGRAMS':
        try {
            if (!isset($data['account_id'])) {
                throw new Exception('Campo "account_id" é obrigatório para LIST_PROGRAMS.');
            }
            $account_id = (int)$data['account_id'];

            $stmt = $pdo->prepare("
                SELECT * 
                  FROM loyalty_programs 
                 WHERE account_id = ? 
                   AND status <> 'DELETED'
            ");
            $stmt->execute([$account_id]);
            $programs = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $response = [
                'success'  => true,
                'programs' => $programs
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 4) CREATE_PROGRAM - Criar um novo programa de fidelidade
     */
    case 'CREATE_PROGRAM':
        try {
            if (!isset($data['account_id'], $data['store_id'], $data['program_type'], $data['program_name'])) {
                throw new Exception(
                    'Campos "account_id", "store_id", "program_type" e "program_name" são obrigatórios para CREATE_PROGRAM.'
                );
            }
            $account_id    = (int)$data['account_id'];
            $store_id      = (int)$data['store_id'];
            $program_type  = $data['program_type'];
            $program_name  = $data['program_name'];

            // Campos opcionais
            $program_description        = isset($data['program_description'])        ? $data['program_description'] : '';
            $start_date                = isset($data['start_date'])                 ? $data['start_date'] : null;
            $end_date                  = isset($data['end_date'])                   ? $data['end_date'] : null;
            $status                    = isset($data['status'])                     ? $data['status'] : 'ACTIVE';
            $is_indefinite             = isset($data['is_indefinite'])              ? (int)$data['is_indefinite'] : 0;
            $affiliates_enabled        = isset($data['affiliates_enabled'])         ? (int)$data['affiliates_enabled'] : 0;
            $affiliates_commission_type= isset($data['affiliates_commission_type']) ? $data['affiliates_commission_type'] : 'TPV';
            $affiliates_commission_value=isset($data['affiliates_commission_value'])? (float)$data['affiliates_commission_value'] : 0.00;
            $affiliates_minimum_sale   = isset($data['affiliates_minimum_sale'])    ? (float)$data['affiliates_minimum_sale'] : null;
            $customers_enabled         = isset($data['customers_enabled'])          ? (int)$data['customers_enabled'] : 0;
            $customers_reward_type     = isset($data['customers_reward_type'])      ? $data['customers_reward_type'] : 'DISCOUNT';
            $customers_reward_value    = isset($data['customers_reward_value'])     ? (float)$data['customers_reward_value'] : 0.00;
            $customers_min_purchase    = isset($data['customers_min_purchase'])     ? (float)$data['customers_min_purchase'] : null;
            $points_enabled            = isset($data['points_enabled'])             ? (int)$data['points_enabled'] : 0;
            $points_ratio              = isset($data['points_ratio'])               ? (float)$data['points_ratio'] : 1.00;
            $points_bonus              = isset($data['points_bonus'])               ? (float)$data['points_bonus'] : 0.00;
            $points_extra              = isset($data['points_extra'])               ? $data['points_extra'] : null;

            $stmt = $pdo->prepare("
                INSERT INTO loyalty_programs (
                    account_id, store_id, program_type, program_name, program_description,
                    start_date, end_date, status, is_indefinite,
                    affiliates_enabled, affiliates_commission_type, affiliates_commission_value, affiliates_minimum_sale,
                    customers_enabled, customers_reward_type, customers_reward_value, customers_min_purchase,
                    points_enabled, points_ratio, points_bonus, points_extra,
                    created_at, updated_at
                ) VALUES (
                    :account_id, :store_id, :program_type, :program_name, :program_description,
                    :start_date, :end_date, :status, :is_indefinite,
                    :affiliates_enabled, :affiliates_commission_type, :affiliates_commission_value, :affiliates_minimum_sale,
                    :customers_enabled, :customers_reward_type, :customers_reward_value, :customers_min_purchase,
                    :points_enabled, :points_ratio, :points_bonus, :points_extra,
                    NOW(), NOW()
                )
            ");
            $stmt->execute([
                ':account_id'                 => $account_id,
                ':store_id'                   => $store_id,
                ':program_type'               => $program_type,
                ':program_name'               => $program_name,
                ':program_description'        => $program_description,
                ':start_date'                 => $start_date,
                ':end_date'                   => $end_date,
                ':status'                     => $status,
                ':is_indefinite'              => $is_indefinite,
                ':affiliates_enabled'         => $affiliates_enabled,
                ':affiliates_commission_type' => $affiliates_commission_type,
                ':affiliates_commission_value'=> $affiliates_commission_value,
                ':affiliates_minimum_sale'    => $affiliates_minimum_sale,
                ':customers_enabled'          => $customers_enabled,
                ':customers_reward_type'      => $customers_reward_type,
                ':customers_reward_value'     => $customers_reward_value,
                ':customers_min_purchase'     => $customers_min_purchase,
                ':points_enabled'             => $points_enabled,
                ':points_ratio'               => $points_ratio,
                ':points_bonus'               => $points_bonus,
                ':points_extra'               => $points_extra
            ]);
            $programId = $pdo->lastInsertId();

            $response = [
                'success'   => true,
                'message'   => 'Programa criado com sucesso',
                'program_id'=> $programId
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 5) DETAIL_PROGRAM - Obter detalhes completos de um programa
     */
    case 'DETAIL_PROGRAM':
        try {
            if (!isset($data['program_id'])) {
                throw new Exception('Campo "program_id" é obrigatório para DETAIL_PROGRAM.');
            }
            $program_id = (int)$data['program_id'];

            $stmt = $pdo->prepare("SELECT * FROM loyalty_programs WHERE id = ?");
            $stmt->execute([$program_id]);
            $program = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$program) {
                throw new Exception('Programa não encontrado.');
            }
            $response = [
                'success' => true,
                'program' => $program
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 6) UPDATE_PROGRAM - Atualizar dados de um programa
     */
    case 'UPDATE_PROGRAM':
        try {
            if (!isset($data['program_id'])) {
                throw new Exception('Campo "program_id" é obrigatório para UPDATE_PROGRAM.');
            }
            $program_id = (int)$data['program_id'];

            // Campos permitidos para atualizar (exemplo)
            $allowed = [
                'program_name', 'program_description', 'start_date', 'end_date',
                'program_type', 'store_id', 'status'
            ];

            $fields = [];
            $params = [];
            foreach ($allowed as $field) {
                if (isset($data[$field])) {
                    $fields[]        = "$field = :$field";
                    $params[":$field"] = $data[$field];
                }
            }
            if (empty($fields)) {
                throw new Exception('Nenhum campo para atualizar foi fornecido.');
            }
            $params[':program_id'] = $program_id;

            $sql = "
                UPDATE loyalty_programs
                   SET " . implode(", ", $fields) . ", updated_at = NOW()
                 WHERE id = :program_id
            ";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            $response = [
                'success' => true,
                'message' => 'Programa atualizado com sucesso.'
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 7) DELETE_PROGRAM - Exclusão lógica (status = 'DELETED')
     */
    case 'DELETE_PROGRAM':
        try {
            if (!isset($data['program_id'])) {
                throw new Exception('Campo "program_id" é obrigatório para DELETE_PROGRAM.');
            }
            $program_id = (int)$data['program_id'];

            $stmt = $pdo->prepare("
                UPDATE loyalty_programs
                   SET status = 'DELETED', updated_at = NOW()
                 WHERE id = ?
            ");
            $stmt->execute([$program_id]);

            $response = [
                'success' => true,
                'message' => 'Programa excluído logicamente (status=DELETED).'
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 8) GET_PROGRAM_DETAIL - Obter detalhes de um programa (espelhado do DETAIL_PROGRAM)
     *    (Pode ser que em sua lógica existam pequenas diferenças)
     */
    case 'GET_PROGRAM_DETAIL':
        try {
            if (!isset($data['program_id'])) {
                throw new Exception('Campo "program_id" é obrigatório para GET_PROGRAM_DETAIL.');
            }
            $program_id = (int)$data['program_id'];

            $stmt = $pdo->prepare("SELECT * FROM loyalty_programs WHERE id = ?");
            $stmt->execute([$program_id]);
            $program = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$program) {
                throw new Exception('Programa não encontrado.');
            }
            $response = [
                'success' => true,
                'program' => $program
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 9) GET_TRANSACTION_ITEMS - Obter itens de uma transação
     */
    case 'GET_TRANSACTION_ITEMS':
        try {
            if (!isset($data['transaction_id'])) {
                throw new Exception('Campo "transaction_id" é obrigatório para GET_TRANSACTION_ITEMS.');
            }
            $transaction_id = (int)$data['transaction_id'];

            $stmt = $pdo->prepare("
                SELECT * 
                  FROM loyalty_transaction_items 
                 WHERE transaction_id = ?
            ");
            $stmt->execute([$transaction_id]);
            $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $response = [
                'success' => true,
                'items'   => $items
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 10) FILTER_TRANSACTIONS - Consulta de transações com parâmetros opcionais
     */
    case 'FILTER_TRANSACTIONS':
        try {
            if (!isset($data['account_id'])) {
                throw new Exception('Campo "account_id" é obrigatório para FILTER_TRANSACTIONS.');
            }
            $account_id = (int)$data['account_id'];

            // Número de dias para filtrar (padrão: 7)
            $days = isset($data['days']) ? (int)$data['days'] : 7;

            // Monta a query base (exemplo):
            $query = "
                SELECT lt.* 
                  FROM loyalty_transactions lt
                 INNER JOIN loyalty_programs lp ON lt.loyalty_id = lp.id
                 WHERE lp.account_id = ? 
                   AND lt.transaction_date >= DATE_SUB(NOW(), INTERVAL ? DAY)
            ";
            $params = [$account_id, $days];

            // Filtros opcionais
            if (!empty($data['consumer_cpf'])) {
                $query .= " AND lt.consumer_cpf = ? ";
                $params[] = $data['consumer_cpf'];
            }
            if (!empty($data['consumer_cnpj'])) {
                $query .= " AND lt.consumer_cnpj = ? ";
                $params[] = $data['consumer_cnpj'];
            }
            if (!empty($data['consumer_nome_razao'])) {
                $query .= " AND lt.consumer_nome_razao LIKE ? ";
                $params[] = '%' . $data['consumer_nome_razao'] . '%';
            }
            if (!empty($data['transaction_date'])) {
                $query .= " AND DATE(lt.transaction_date) = ? ";
                $params[] = $data['transaction_date'];
            }

            $query .= " ORDER BY lt.transaction_date DESC";

            $stmt = $pdo->prepare($query);
            $stmt->execute($params);
            $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $response = [
                'success'      => true,
                'transactions' => $transactions
            ];
            logApiCall($pdo, $account_id, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 11) LIST_FILTER_TRANSACTIONS - Listar dados agregados de transações
     */
    case 'LIST_FILTER_TRANSACTIONS':
        try {
            if (!isset($data['account_id'])) {
                throw new Exception('Campo "account_id" é obrigatório para LIST_FILTER_TRANSACTIONS.');
            }
            $store_account_id = (int)$data['account_id'];

            // Filtros opcionais
            $cpf         = isset($data['cpf'])          ? $data['cpf'] : null;
            $cnpj        = isset($data['cnpj'])         ? $data['cnpj'] : null;
            $razaoSocial = isset($data['razao_social']) ? $data['razao_social'] : null;
            $programa    = isset($data['programa'])     ? $data['programa'] : null;
            $dataFiltro  = isset($data['data'])         ? $data['data'] : null;

            // Paginação (exemplo: 30 registros por página)
            $page   = isset($data['page']) ? (int)$data['page'] : 1;
            $limit  = 30;
            $offset = ($page - 1) * $limit;

            $params = [];
            $sql = "
                SELECT
                    a.id AS beneficiary_account_id,
                    COALESCE(ap.full_name, ab.business_name) AS consumer_nome_razao,
                    a.type AS account_type,
                    ap.cpf,
                    ab.cnpj,
                    lp.program_type,
                    lp.id AS id_program,
                    lp.program_name,
                    lp.start_date,
                    lp.end_date,
                    lp.is_indefinite,
                    -- Totais EARN disponíveis (vencidos há mais de 8 dias)
                    SUM(CASE WHEN lt.transaction_type = 'EARN'
                              AND DATEDIFF(CURDATE(), DATE(lt.created_at)) > 8 
                             THEN lt.amount ELSE 0 END) AS total_transactions_available,
                    -- Totais EARN em processamento (menos de 8 dias)
                    SUM(CASE WHEN lt.transaction_type = 'EARN'
                              AND DATEDIFF(CURDATE(), DATE(lt.created_at)) <= 8 
                             THEN lt.amount ELSE 0 END) AS total_transactions_processing,
                    -- Total já resgatado (REDEEM)
                    SUM(CASE WHEN lt.transaction_type = 'REDEEM'
                             THEN lt.amount ELSE 0 END) AS total_redeemed,
                    
                    -- Exemplo de cálculo de cashback, points ou miles disponível, etc.
                    CASE 
                      WHEN lp.program_type = 'cashback' 
                           AND SUM(CASE WHEN lt.transaction_type='EARN'
                                        AND DATEDIFF(CURDATE(), lt.created_at) > 8
                                        THEN lt.amount ELSE 0 END) >= lp.customers_min_purchase
                      THEN SUM(CASE WHEN lt.transaction_type='EARN'
                                    AND DATEDIFF(CURDATE(), lt.created_at) > 8
                                    THEN lt.amount ELSE 0 END) 
                           * lp.customers_reward_value / 100
                      ELSE 0
                    END AS cash_back_available,

                    -- Afiliados, etc...
                    CASE
                      WHEN lp.affiliates_enabled = 1 
                           AND SUM(CASE WHEN lt.transaction_type = 'EARN' 
                                        AND DATEDIFF(CURDATE(), lt.created_at) > 8 
                                        THEN lt.amount ELSE 0 END) >= lp.affiliates_minimum_sale
                      THEN CASE 
                             WHEN lp.affiliates_commission_type = 'FIXED'
                               THEN COUNT(CASE WHEN lt.transaction_type='EARN' AND DATEDIFF(CURDATE(), lt.created_at) > 8 THEN 1 END)
                                    * lp.affiliates_commission_value
                             WHEN lp.affiliates_commission_type = 'TPV'
                               THEN SUM(CASE WHEN lt.transaction_type='EARN' AND DATEDIFF(CURDATE(), lt.created_at) > 8 THEN lt.amount ELSE 0 END)
                                    * lp.affiliates_commission_value / 100
                             ELSE 0
                           END
                      ELSE 0
                    END AS affiliates_commission_available

                FROM loyalty_transactions lt
                INNER JOIN loyalty_programs lp ON lt.loyalty_id = lp.id
                INNER JOIN account a ON lt.beneficiary_account_id = a.id
                LEFT JOIN account_personal ap ON a.id = ap.account_id AND a.type = 'PF'
                LEFT JOIN account_business ab ON a.id = ab.account_id AND a.type = 'PJ'
                
                WHERE lp.account_id = ?
                  AND lp.start_date <= CURDATE()
                  AND (lp.end_date >= CURDATE() OR lp.is_indefinite = 1)
            ";
            $params[] = $store_account_id;

            // Filtro de data (data específica) ou últimos 30 dias
            if ($dataFiltro) {
                $dateParts = explode('/', $dataFiltro);
                if (count($dateParts) != 3) {
                    throw new Exception("Data inválida: use o formato dd/mm/yy.");
                }
                $day   = $dateParts[0];
                $month = $dateParts[1];
                $year  = (strlen($dateParts[2]) == 2) ? "20".$dateParts[2] : $dateParts[2];
                $specificDate = "$year-$month-$day";

                $sql .= " AND DATE(lt.created_at) = ?";
                $params[] = $specificDate;
            } else {
                $sql .= " AND lt.created_at BETWEEN DATE_SUB(CURDATE(), INTERVAL 30 DAY) AND CURDATE()";
            }

            // Filtros opcionais
            if ($cpf) {
                $sql .= " AND a.type = 'PF' AND ap.cpf = ?";
                $params[] = $cpf;
            }
            if ($cnpj) {
                $sql .= " AND a.type = 'PJ' AND ab.cnpj = ?";
                $params[] = $cnpj;
            }
            if ($razaoSocial) {
                $sql .= " AND ((a.type = 'PF' AND ap.full_name LIKE ?) OR (a.type = 'PJ' AND ab.business_name LIKE ?))";
                $params[] = "%".$razaoSocial."%";
                $params[] = "%".$razaoSocial."%";
            }
            if ($programa) {
                $sql .= " AND lp.program_name LIKE ?";
                $params[] = "%".$programa."%";
            }

            $sql .= " GROUP BY a.id, lp.id";
            $sql .= " LIMIT " . (int)$limit . " OFFSET " . (int)$offset;

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Ajuste de end_date se is_indefinite == 1
            foreach ($transactions as &$t) {
                if ($t['is_indefinite'] == 1) {
                    $t['end_date'] = "Indeterminado";
                }
            }

            $response = [
                'success'      => true,
                'transactions' => $transactions,
                'page'         => $page
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ['error' => $e->getMessage()];
            error_log($e->getMessage());
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 12) GET_PROGRAM_TRANSACTION_DETAIL - Detalhar transações de um programa
     */
    case 'GET_PROGRAM_TRANSACTION_DETAIL':
        try {
            if (!isset($data['program_id'])) {
                throw new Exception("Campo 'program_id' é obrigatório para GET_PROGRAM_TRANSACTION_DETAIL.");
            }
            $program_id = (int)$data['program_id'];
            if ($program_id <= 0) {
                throw new Exception("program_id deve ser um inteiro positivo.");
            }

            // Parâmetros opcionais
            $type_transaction = isset($data['type_transaction']) ? (int)$data['type_transaction'] : null;
            if ($type_transaction === 0) {
                $type_transaction = null;
            }

            $beneficiary_account_id = isset($data['beneficiary_account_id']) ? (int)$data['beneficiary_account_id'] : null;

            $params = [$program_id];
            $sql = "
                SELECT *
                  FROM loyalty_transactions
                 WHERE loyalty_id = ?
                   AND created_at BETWEEN DATE_SUB(CURDATE(), INTERVAL 120 DAY) AND CURDATE()
            ";

            // Filtra por type_transaction, se informado
            if ($type_transaction !== null) {
                $sql .= " AND transaction_type = ?";
                $params[] = $type_transaction;
            }

            // Filtra por beneficiary_account_id, se informado
            if ($beneficiary_account_id !== null) {
                $sql .= " AND beneficiary_account_id = ?";
                $params[] = $beneficiary_account_id;
            }

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $response = [
                "success"      => true,
                "transactions" => $transactions
            ];
            logApiCall($pdo, 0, $input, json_encode($response), 200);
            sendResponse($response);

        } catch (Exception $e) {
            http_response_code(500);
            $error_response = ["error" => $e->getMessage()];
            error_log("GET_PROGRAM_TRANSACTION_DETAIL >> Erro: " . $e->getMessage());
            logApiCall($pdo, 0, $input, json_encode($error_response), 500);
            sendResponse($error_response);
        }
        break;

    /**
     * 13) LIST_CALCULATE_BONUS_MULTIPLIQUE - Consulta consolidada de bonus
     */
    case 'LIST_CALCULATE_BONUS_MULTIPLIQUE':
        // Parâmetros básicos
        $account_id =               isset($data['account_id'])      ? $data['account_id']       : null;
        $beneficiary_account_id =   isset($data['beneficiary_id'])  ? $data['beneficiary_id']   : null;
        $start_date =               isset($data['start_date'])      ? $data['start_date']       : null;
        $end_date   =               isset($data['end_date'])        ? $data['end_date']         : null;

        if (!$account_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Parâmetro "account_id" é obrigatório.']);
            exit;
        }

        try {
            // Monta a query
            $sql = "
                SELECT 
                    lp.id,
                    lp.program_name,
                    lp.program_type,
                    lp.start_date,
                    lp.end_date,
                    lp.status,
                    lp.store_id,
                    COUNT(*) AS total_transacoes,
                    SUM(
                        CASE 
                            WHEN TIMESTAMPDIFF(DAY, lt.transaction_date, NOW()) >= 8 
                                 THEN lt.amount 
                                 ELSE 0 
                        END
                    ) AS total_faturado,
                    SUM(
                        CASE 
                            WHEN TIMESTAMPDIFF(DAY, lt.transaction_date, NOW()) >= 8 
                                 THEN lt.remaining_amount 
                                 ELSE 0 
                        END
                    ) AS total_bonus_disponivel,
                    SUM(
                        CASE 
                            WHEN TIMESTAMPDIFF(DAY, lt.transaction_date, NOW()) < 8 
                                 THEN lt.amount 
                                 ELSE 0 
                        END
                    ) AS total_bonus_em_processamento,
                    ROUND(
                      SUM(
                        CASE 
                            WHEN lp.customers_reward_type = 'FIXED' 
                                 THEN lp.customers_reward_value
                            WHEN lp.customers_reward_type = 'POINTS'
                                 THEN lp.customers_reward_value
                            WHEN lp.customers_reward_type = 'PERCENT'
                                 THEN (lt.amount / 100) * lp.customers_reward_value
                            ELSE 0
                        END
                      ), 2
                    ) AS total_customers_reward,
                    ROUND(
                      SUM(
                        CASE 
                            WHEN lp.affiliates_enabled = 1 THEN
                                CASE 
                                    WHEN lp.affiliates_commission_type = 'TPV' 
                                        THEN (lt.amount / 100) * lp.affiliates_commission_value
                                    WHEN lp.affiliates_commission_type = 'FIXED'
                                        THEN lp.affiliates_commission_value
                                    ELSE 0
                                END
                            ELSE 0
                        END
                      ), 2
                    ) AS total_affiliates_commission,
        
                    -- Colunas da account
                    a.id AS account_id,
                    a.email AS email,
                    
                    -- Colunas de account_personal
                    ap.cpf AS cpf,
                    ap.full_name AS full_name,
        
                    -- Colunas de account_business
                    ab.business_name AS business_name,
                    ab.cnpj AS cnpj
        
                FROM loyalty_transactions lt
                INNER JOIN loyalty_programs lp ON lt.loyalty_id = lp.id
                INNER JOIN account a ON lt.beneficiary_account_id = a.id
                LEFT JOIN account_personal ap ON ap.account_id = a.id
                LEFT JOIN account_business ab ON ab.account_id = lp.account_id
                WHERE 
                    lt.transaction_type = 'EARN'
                    AND lp.status = 'ACTIVE'
                    AND NOW() BETWEEN lp.start_date AND lp.end_date
                    AND a.id = :account_id
            ";

            $params = [':account_id' => $account_id];

            if ($account_id) {
                $sql .= " AND lp.account_id = :account_id";
                $params[':account_id'] = $account_id;
            }
            if ($beneficiary_account_id) {
                $sql .= " AND lt.beneficiary_account_id = :beneficiary_account_id";
                $params[':beneficiary_account_id'] = $beneficiary_account_id;
            }
            if ($start_date) {
                $sql .= " AND lt.transaction_date >= :start_date";
                $params[':start_date'] = $start_date;
            }
            if ($end_date) {
                $sql .= " AND lt.transaction_date <= :end_date";
                $params[':end_date'] = $end_date;
            }

            $sql .= " GROUP BY lp.id";

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode($result, JSON_PRETTY_PRINT);

        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao consultar o banco de dados: ' . $e->getMessage()]);
            exit;
        }
        break;

    // ---------------------------------------------------
    // Ação padrão se não corresponder a nenhuma acima
    // ---------------------------------------------------
    default:
        http_response_code(400);
        $error_response = ['error' => 'Ação inválida. Verifique o campo "action" e tente novamente.'];
        logApiCall($pdo, 0, $input, json_encode($error_response), 400);
        sendResponse($error_response);
        break;
}

?>
